//*************************************************************************************************
//
//  Description:
//    ZPrimeColour.fx - Shaders for rendering colour + depth.
//
//    Copyright (c) 2008 MWL. All rights reserved.
//
//*************************************************************************************************

#include "stddefs.fxh"

//-----------------------------------------------------------------------
//
// Input parameters
//

#if defined( _3DSMAX_ ) || defined(USE_WVP_CONSTANT)
// Max doesn't support viewproj as an app-supplied parameter
float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;


// The fill colour used by ZPrimeColour_Constant technique.
float4 constantColour
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = true;
> = { 1.0f, 0.0f, 0.0f, 1.0f };


// Texture used by ZPrimeColour_Texture technique.
texture constantTexture : TEXTURE
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = true;
>;

// Sampler used by ZPrimeColour_Texture technique.
sampler2D constantSampler : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="constantTexture"; 
	string MinFilter = "Point";
	string MagFilter = "Point";
	string MipFilter = "None";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < constantTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Point;
	MagFilter = Point;
	MipFilter = None;
	AddressU  = Clamp;
	AddressV  = Clamp;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
#endif
};


// Transformation from world space into cubemap space (ZPrimeColour_CubeMap technique).
float4x4 worldToCube
<
	string UIWidget = "None";
	bool appEdit = true;
	bool export = false;
>;


// Texture used by ZPrimeColour_CubeMap technique.
texture cubeMapTexture : TEXTURE
<
	string UIWidget = "None";
	bool export = false;
	bool appEdit = true;
>;

// Sampler used by ZPrimeColour_CubeMap technique.
samplerCUBE cubeMapSampler : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="cubeMapTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	string AddressW  = "Clamp";
	int MipMapLODBias = 0;
>
= sampler_state
{
	Texture = < cubeMapTexture >;
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = Linear;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	AddressW  = ClampToEdge;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	AddressW  = Clamp;
#endif
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
};


//-----------------------------------------------------------------------
//
// Vertex Shader
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;														// Object space position
};

// Output structures
struct VSOUTPUT
{
	float4 position		: POSITION;													// View-coords position
};


struct VSOUTPUT_CubeMap
{
	float4 position		: POSITION;													// View-coords position
	float4 coords		: TEXCOORD0;													// Cubemap coordinate system vertex position.
};


VSOUTPUT ZPrimeColourVertexShader( float3 pos : POSITION )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( pos, 1.0f ), worldviewproj );

	return _output;
}


VSOUTPUT_CubeMap ZPrimeColourVertexShader_CubeMap( float3 pos : POSITION )
{
	VSOUTPUT_CubeMap _output;

#if !defined( _3DSMAX_ ) && !defined(USE_WVP_CONSTANT)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( pos, 1.0f ), worldviewproj );

	// Calculate cubemap-space position of the vertex.
	// Convert from object space into world space, and then into cubemap space -
	// it is the most convenient way, while not the fastest.
	float4 wsPosition = mul( float4( pos, 1.0f ), world );
	_output.coords = mul( wsPosition, worldToCube );

	return _output;
}




//-----------------------------------------------------------------------
//
// Fragment Shaders
//

COLOUR_OUTPUT_TYPE ZPrimeColourFragmentShader_Constant() : COLOR0
{
	COLOUR_OUTPUT_TYPE result = CalculateOutputPixel( constantColour );
	// Ensure we output 0 into alpha - even on PS3, we do not want luminance there.
	// The black alpha is used as mask, detected during scene luminance computation.
	//result.a = 0.0f;
	return result;
}

COLOUR_OUTPUT_TYPE ZPrimeColourFragmentShader_Texture() : COLOR0
{
	COLOUR_OUTPUT_TYPE result = CalculateOutputPixel( tex2D( constantSampler, float2( 0.5f, 0.5f ) ) );
	// Ensure we output 0 into alpha - even on PS3, we do not want luminance there.
	// The black alpha is used as mask, detected during scene luminance computation.
	//result.a = 0.0f;
	return result;
}

COLOUR_OUTPUT_TYPE ZPrimeColourFragmentShader_CubeMap( float4 coords : TEXCOORD0 ) : COLOR0
{
	COLOUR_OUTPUT_TYPE result = CalculateOutputPixel( texCUBE( cubeMapSampler, coords.xyz ) );
	// Ensure we output 0 into alpha - even on PS3, we do not want luminance there.
	// The black alpha is used as mask, detected during scene luminance computation.
	//result.a = 0.0f;
	return result;
}


//-----------------------------------------------------------------------
//
// Technique(s)
//


technique ZPrimeColour_Constant
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string zprimeBehaviour	= "ERMB_RENDER";
	string zprimeTechnique	= "ZPrimeColour_Constant";
	int    zprimeDeferredID	= 0;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx ZPrimeColourVertexShader();
		PixelShader = compile sce_fp_rsx ZPrimeColourFragmentShader_Constant();
#else
		VertexShader = compile vs_3_0 ZPrimeColourVertexShader();
		PixelShader = compile ps_3_0 ZPrimeColourFragmentShader_Constant();
#endif
	}
}


technique ZPrimeColour_Texture
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string zprimeBehaviour	= "ERMB_RENDER";
	string zprimeTechnique	= "ZPrimeColour_Texture";
	int    zprimeDeferredID	= 0;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx ZPrimeColourVertexShader();
		PixelShader = compile sce_fp_rsx ZPrimeColourFragmentShader_Texture();
#else
		VertexShader = compile vs_3_0 ZPrimeColourVertexShader();
		PixelShader = compile ps_3_0 ZPrimeColourFragmentShader_Texture();
#endif
	}
}


technique ZPrimeColour_CubeMap
<
	bool supportsSpecialisedLighting = false;
  bool preservesGlobalState = true;
	string zprimeBehaviour	= "ERMB_RENDER";
	string zprimeTechnique	= "ZPrimeColour_CubeMap";
	int    zprimeDeferredID	= 0;
>
{
	pass Pass0
	{
#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx ZPrimeColourVertexShader_CubeMap();
		PixelShader = compile sce_fp_rsx ZPrimeColourFragmentShader_CubeMap();
#else
		VertexShader = compile vs_3_0 ZPrimeColourVertexShader_CubeMap();
		PixelShader = compile ps_3_0 ZPrimeColourFragmentShader_CubeMap();
#endif
	}
}
